/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtCore module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

/****************************************************************************
**
** Copyright (C) 2023-2025 Linaro Limited (or its affiliates). All rights reserved.
** Copyright (C) 2012-2023 Arm Limited (or its affiliates).
**
****************************************************************************/
#include "qpipereader_p.h"
#include <io.h>
#include <errno.h>
#if !defined EWOULDBLOCK
#define EWOULDBLOCK 140
#endif

Q_PIPE qt_open_read_pipe(const char* pathname)
{
    return ::CreateNamedPipeA(pathname, 
                              PIPE_ACCESS_INBOUND, 
                              PIPE_NOWAIT, 
                              PIPE_UNLIMITED_INSTANCES,
                              1024, 
                              1024, 
                              3000, 
                              NULL);
}

qint64 qt_native_read(Q_PIPE pipe, char* data, qint64 maxlen)
{
    DWORD bytesRead = 0;
    ::ReadFile(pipe, data, static_cast<DWORD>(maxlen), &bytesRead, 0);
    return bytesRead;
}

void QPipeReaderPrivate::closePipe()
{
    if (pipe != INVALID_Q_PIPE)
    {
        ::CloseHandle(pipe);
        pipe = INVALID_Q_PIPE;
    }
}

void QPipeReaderPrivate::createNotifier()
{
}

qint64 QPipeReaderPrivate::bytesAvailable() const
{
    DWORD bytesAvail = 0;
	::PeekNamedPipe(pipe, 0, 0, 0, &bytesAvail, 0);

#if defined (QPIPEWRITER_DEBUG)
    qDebug("QPipeWriterPrivate::bytesAvailable() == %ld", bytesAvail);
#endif
    return bytesAvail;
}

/*! \internal
*/
bool QPipeReaderPrivate::_q_canRead(bool drain)
{
    Q_Q(QPipeReader);
    qint64 available;
    if (blocking) {
        available = bytesAvailable();
        if (available == 0) 
        {
            if (drain) 
            {
                // 0 bytes available is not a problem if called from drain.
                return false;
            }
        } 
        else if (available == -1) 
        {
            available = 1;
        }
    } 
    else 
    {
        available = 4096;
    }

    qint64 totalReadBytes = 0;
    do {
        char *ptr = buffer.reserve(available);
        qint64 readBytes = read(ptr, available);
        if (readBytes == -1) {
            if (errno == EAGAIN || errno == EWOULDBLOCK) {
                buffer.chop(available);
                break;
            } else if (errno == EBADF) {
                // Treat EBADF as EOF.
                readBytes = 0;
            } else {
                q->setErrorString(QPipeReader::tr("read error"));
                emit q->readError();
#if defined QPIPEREADER_DEBUG
                qDebug("QPipeReaderPrivate::canRead(), failed to read from the pipe");
#endif
                return false;
            }
        }
#if defined QPIPEREADER_DEBUG
        qDebug("QPipeReaderPrivate::canRead, read %d bytes from the pipe",
                int(readBytes));
#endif

        buffer.chop(available - readBytes);

        if (readBytes == 0) {
            eof = true;
#if defined QPIPEREADER_DEBUG
            qDebug("QPipeReaderPrivate::canRead(), end of file");
#endif
            emit q->eof();
            break;
        } else if (logFile) {
            logFile->write(ptr, readBytes);
            logFile->flush();
        }
        totalReadBytes += readBytes;
    } while (!blocking);
    if (!drain)
        emit q->readyRead();
    return totalReadBytes > 0;
}
