/**************************************************************************
**
** This file is part of Qt Creator
**
** Copyright (c) 2011 Nokia Corporation and/or its subsidiary(-ies).
**
** Contact: Nokia Corporation (info@qt.nokia.com)
**
**
** GNU Lesser General Public License Usage
**
** This file may be used under the terms of the GNU Lesser General Public
** License version 2.1 as published by the Free Software Foundation and
** appearing in the file LICENSE.LGPL included in the packaging of this file.
** Please review the following information to ensure the GNU Lesser General
** Public License version 2.1 requirements will be met:
** http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** Other Usage
**
** Alternatively, this file may be used in accordance with the terms and
** conditions contained in a signed written agreement between you and Nokia.
**
** If you have questions regarding the use of this file, please contact
** Nokia at info@qt.nokia.com.
**
**************************************************************************/

#include "detailswidget.h"
#include "detailsbutton.h"
#include "detailswidget_p.h"

#include <QStack>
#include <QPropertyAnimation>
#include <QDebug>

#include <QGridLayout>
#include <QLabel>
#include <QCheckBox>
#include <QPainter>
#include <QScrollArea>
#include <QApplication>
#include <QMouseEvent>
#include <QPaintEngine>
#include <QColormap>
#include <QDockWidget>

/*!
    \class Utils::DetailsWidget

    \brief Widget a button to expand a 'Details' area.

    This widget is using a grid layout and places the items
    in the following way:

    \code
+------------+-------------------------+---------------+
+summaryLabel|              toolwidget | detailsButton |
+------------+-------------------------+---------------+
+                additional summary                    |
+------------+-------------------------+---------------+
|                  widget                              |
+------------+-------------------------+---------------+
    \endcode
*/

static const int MARGIN = 6;

//! Extension to QLabel that can emit a signal when its background is clicked
/*! This class adds the \c labelBackgroundClicked signal, emitted when
 *  the label's background (any portion of the label that isn't a hyperlink)
 *  is clicked. Clicking on a hyperlink in the label will cause a
 *  \c linkActivated signal to be emitted as normal. A \c labelBackgroundClicked
 *  signal will \e not be emitted if a hyperlink is clicked. */
class DetailsWidgetLabel : public QLabel
{
    Q_OBJECT
public:
    DetailsWidgetLabel(QWidget * parent = 0, Qt::WindowFlags f = Qt::WindowFlags())
        : QLabel(parent, f) {
        connect(this, &DetailsWidgetLabel::linkActivated,
                this, &DetailsWidgetLabel::on_linkActivated);
    }
    DetailsWidgetLabel(const QString& text, QWidget* parent = nullptr, Qt::WindowFlags f = Qt::WindowFlags())
        : QLabel(text, parent, f) {
        connect(this, &DetailsWidgetLabel::linkActivated,
                this, &DetailsWidgetLabel::on_linkActivated);
    }

signals:
    //! Emitted when a non-hyperlink portion of the label is clicked
    void labelBackgroundClicked();

protected:
    virtual void mouseReleaseEvent(QMouseEvent* ev) override {
        mLinkActivatedSignalEmitted = false;
        QLabel::mouseReleaseEvent(ev);
        if (!mLinkActivatedSignalEmitted) {
            // A hyperlink wasn't click. Emit signal informing
            // of a non-link mouse click on the label.
            emit labelBackgroundClicked();
        }
    }

private slots:
    void on_linkActivated(const QString&) {
        mLinkActivatedSignalEmitted = true;
    }

private:
    bool mLinkActivatedSignalEmitted;
};

class DetailsWidgetPrivate
{
public:
    DetailsWidgetPrivate(QWidget *parent);

    QPixmap cacheBackground(const QSize &size, bool expanded);
    void updateControls();
    void changeHoverState(bool hovered);

    QWidget *q;
    DetailsButton *m_detailsButton;
    QGridLayout *m_grid;
    DetailsWidgetLabel *m_summaryLabel;
    QCheckBox *m_summaryCheckBox;
    QLabel *m_additionalSummaryLabel;
    QWidget *m_toolWidget;
    QWidget *m_widget;
    QLayoutItem *m_spacer;

    QPixmap m_collapsedPixmap;
    QPixmap m_expandedPixmap;

    DetailsWidget::State m_state;
    bool m_hovered;
    bool m_useCheckBox;
    bool m_detailsButtonVisible;
    bool m_detailsButtonEnabled;
    bool m_useGradient;
    bool m_useTransparency;
    bool m_toggleDetailsOnSummaryBarClicks; //!< Does clicking on summary line/bar toggle the visiblilty of the 'details' portion of the widget
};

DetailsWidgetPrivate::DetailsWidgetPrivate(QWidget *parent) :
        q(parent),
        m_detailsButton(new DetailsButton(parent)),
        m_grid(new QGridLayout),
        m_summaryLabel(new DetailsWidgetLabel(parent)),
        m_summaryCheckBox(new QCheckBox(parent)),
        m_additionalSummaryLabel(new QLabel(parent)),    
        m_toolWidget(0),
        m_widget(0),
        m_spacer(new QSpacerItem(MARGIN, MARGIN, QSizePolicy::Fixed, QSizePolicy::Fixed)),
        m_state(DetailsWidget::Collapsed),
        m_hovered(false),
        m_useCheckBox(false),
        m_detailsButtonVisible(true),
        m_detailsButtonEnabled(true),
        m_toggleDetailsOnSummaryBarClicks(true)
{
    m_summaryLabel->setTextInteractionFlags(Qt::LinksAccessibleByMouse);
    m_summaryLabel->setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Fixed);
    m_summaryLabel->setContentsMargins(MARGIN, MARGIN, MARGIN, MARGIN);
    //m_summaryLabel->setWordWrap(true);

    m_summaryCheckBox->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
    m_summaryCheckBox->setContentsMargins(MARGIN, MARGIN, MARGIN, MARGIN);
    //m_summaryCheckBox->setAttribute(Qt::WA_LayoutUsesWidgetRect); /* broken layout on mac otherwise */
    m_summaryCheckBox->setVisible(false);

    m_additionalSummaryLabel->setTextInteractionFlags(Qt::TextSelectableByMouse);
    m_additionalSummaryLabel->setContentsMargins(MARGIN, MARGIN, MARGIN, MARGIN);
    m_additionalSummaryLabel->setWordWrap(true);
    m_additionalSummaryLabel->setVisible(false);

    m_grid->setContentsMargins(0, 0, 0, 0);
    m_grid->setSpacing(0);
    m_grid->addWidget(m_summaryLabel, 0, 2);
    m_grid->addWidget(m_detailsButton, 0, 4);
    m_grid->addWidget(m_additionalSummaryLabel, 1, 0, 1, 5);

    QWidget::setTabOrder(m_summaryCheckBox, m_detailsButton);

    m_useTransparency = QColormap::instance().depth() >= 16;
    m_useGradient = QColormap::instance().depth() >= 16;
}

QPixmap DetailsWidgetPrivate::cacheBackground(const QSize &size, bool /* expanded */)
{
    QPixmap pixmap(size);
    QColor bg;
    if(m_useTransparency) {
        pixmap.fill(Qt::transparent);
    } else {
        bg = q->palette().color(q->backgroundRole());
        pixmap.fill(bg);
    }
    QPainter p(&pixmap);

    int topHeight = qMax(qMax(m_detailsButton->height(), m_summaryLabel->height()), m_useCheckBox ? m_summaryCheckBox->height() : 0);
    QRect topRect(0, 0, size.width(), topHeight);
    QRect fullRect(0, 0, size.width(), size.height());
#ifdef Q_OS_MACOS
    // TODO: should be using autoFillBackground if this is really necessary!
    //p.fillRect(fullRect, qApp->palette().window().color());
#endif
    if (m_useTransparency)
        p.fillRect(fullRect, QColor(255, 255, 255, 40));
    else
        p.fillRect(fullRect, mix(QColor(255, 255, 255, 40), bg));

#if 0
    QColor highlight = q->palette().highlight().color();
    highlight.setAlpha(0.5);
    if (expanded)
        p.fillRect(topRect, highlight);
#endif

    if (m_useGradient) {
        QLinearGradient lg(topRect.topLeft(), topRect.bottomLeft());
        if (m_useTransparency) {
            lg.setColorAt(1, QColor(255, 255, 255, 0));
            lg.setColorAt(0, QColor(255, 255, 255, 130));
        } else {
            QColor fillColour(mix(QColor(255, 255, 255, 40), bg));
            lg.setColorAt(1, mix(QColor(255, 255, 255, 0), fillColour));
            lg.setColorAt(0, mix(QColor(255, 255, 255, 130), fillColour));
        }
        p.fillRect(topRect, lg);
    }
    p.setRenderHint(QPainter::Antialiasing, true);
    p.translate(0.5, 0.5);
    if (m_useTransparency)
        p.setPen(QColor(0, 0, 0, 40));
    else
        p.setPen(mix(QColor(0, 0, 0, 40), bg));
    p.setBrush(Qt::NoBrush);
    p.drawRoundedRect(fullRect.adjusted(0, 0, -1, -1), 2, 2);
    p.setBrush(Qt::NoBrush);
    if (m_useTransparency)
        p.setPen(QColor(255,255,255,140));
    else
        p.setPen(mix(QColor(255,255,255,140), bg));
    p.drawRoundedRect(fullRect.adjusted(1, 1, -2, -2), 2, 2);
    p.setPen(QPen(q->palette().color(QPalette::Mid)));

    return pixmap;
}

void DetailsWidgetPrivate::updateControls()
{
    if (m_widget)
        m_widget->setVisible((!m_useCheckBox || m_summaryCheckBox->isChecked()) &&
                             (m_state == DetailsWidget::Expanded || m_state == DetailsWidget::NoSummary));
    m_detailsButton->setChecked(m_state == DetailsWidget::Expanded && m_widget);
    m_detailsButton->setEnabled(m_detailsButtonEnabled && (!m_useCheckBox || m_summaryCheckBox->isChecked()));
    //m_summaryLabel->setEnabled(m_state == DetailsWidget::Collapsed && m_widget);
    m_detailsButton->setVisible(m_state != DetailsWidget::NoSummary && m_detailsButtonVisible);
    m_summaryLabel->setVisible(m_state != DetailsWidget::NoSummary);
    m_summaryCheckBox->setVisible(m_state != DetailsWidget::NoSummary && m_useCheckBox);
    if(m_toolWidget)
        m_toolWidget->setEnabled(!m_useCheckBox || m_summaryCheckBox->isChecked());

    for (QWidget *w = q; w; w = w->parentWidget()) {

        // If a parent is a QDockWidget it may not get laid out / resize correctly
        // when m_widget is shown / hidden. For some unknown reason setting
        // the dock widget's minimum height to 0 here can avoid this problem.
        //
        // If there are instances where this doesn't work, the alternative solution
        // is to:
        // * calculate the change in height caused by the above commands
        //   (take sizeHint().height() before and after)
        // * force the dockwidget to change size by the same amount (set its minimum
        //   & maximum height to that target size)
        // * force an event loop to process using qApp->processEvents()
        // * restore the dock widget's ability to be resized by the user by setting
        //   the minimum height of the dock widget to 0, and the max to QWIDGETSIZE_MAX.
        //   Restoring the original values can cause the widget to not resize
        //   correctly.
        if (QDockWidget *dock = qobject_cast<QDockWidget*>(w)) {
            dock->setMinimumHeight(0);
        }

        if (w->layout())
            w->layout()->activate();

        if (QScrollArea *area = qobject_cast<QScrollArea*>(w)) {
            QEvent e(QEvent::LayoutRequest);
            QCoreApplication::sendEvent(area, &e);
        }
    }
}

void DetailsWidgetPrivate::changeHoverState(bool hovered)
{
    if (!m_toolWidget)
        return;
    m_hovered = hovered;
}


DetailsWidget::DetailsWidget(QWidget *parent) :
        QWidget(parent),
        d(new DetailsWidgetPrivate(this))
{
    setLayout(d->m_grid);

    connect(d->m_detailsButton, &DetailsButton::toggled,
            this, &DetailsWidget::setExpanded);
    connect(d->m_detailsButton, &DetailsButton::clicked,
            this, &DetailsWidget::detailsClicked);
    connect(d->m_summaryCheckBox, &QCheckBox::toggled,
            this, &DetailsWidget::checked);
    connect(d->m_summaryCheckBox, &QCheckBox::toggled,
            this, &DetailsWidget::summaryCheckBoxToggled);
    connect(d->m_summaryLabel, &DetailsWidgetLabel::linkActivated,
            this, &DetailsWidget::linkActivated);
    connect(d->m_summaryLabel, &DetailsWidgetLabel::labelBackgroundClicked,
            this, &DetailsWidget::on_labelBackgroundClicked);
    d->m_summaryLabel->installEventFilter(this);
    d->updateControls();
}

DetailsWidget::~DetailsWidget()
{
    delete d;
}

bool DetailsWidget::useCheckBox()
{
    return d->m_useCheckBox;
}

void DetailsWidget::setUseCheckBox(bool b)
{
    if (d->m_useCheckBox == b)
        return;
    d->m_useCheckBox = b;
    if (!b) {
        d->m_grid->removeItem(d->m_spacer);
        d->m_grid->removeWidget(d->m_summaryCheckBox);
    } else {
        d->m_grid->addItem(d->m_spacer, 0, 0);
        d->m_grid->addWidget(d->m_summaryCheckBox, 0, 1);
    }
    d->m_summaryCheckBox->setVisible(b);
    d->updateControls();
}

void DetailsWidget::setCheckBoxEnabled(bool b)
{
    d->m_summaryCheckBox->setEnabled(b);
}

bool DetailsWidget::checkBoxEnabled() const
{
    return d->m_summaryCheckBox->isEnabled();
}

void DetailsWidget::setChecked(bool b)
{
    d->m_summaryCheckBox->setChecked(b);
}

bool DetailsWidget::isChecked() const
{
    return d->m_useCheckBox && d->m_summaryCheckBox->isChecked();
}

void DetailsWidget::setSummaryFontBold(bool b)
{
    QFont f;
    f.setBold(b);
    d->m_summaryCheckBox->setFont(f);
    d->m_summaryLabel->setFont(f);
}

void DetailsWidget::setIcon(const QIcon &icon)
{
    d->m_summaryCheckBox->setIcon(icon);
}

void DetailsWidget::paintEvent(QPaintEvent *paintEvent)
{
    QWidget::paintEvent(paintEvent);

    const QRect paintArea(contentsRect());
    QPainter p(this);

    if (d->m_state != Expanded) {
        if (d->m_collapsedPixmap.isNull() ||
            d->m_collapsedPixmap.size() != size())
            d->m_collapsedPixmap = d->cacheBackground(paintArea.size(), false);
        p.drawPixmap(paintArea, d->m_collapsedPixmap);
    } else {
        if (d->m_expandedPixmap.isNull() ||
            d->m_expandedPixmap.size() != size())
            d->m_expandedPixmap = d->cacheBackground(paintArea.size(), true);
        p.drawPixmap(paintArea, d->m_expandedPixmap);
    }
}

#if QT_VERSION < QT_VERSION_CHECK(6,0,0)
void DetailsWidget::enterEvent(QEvent * event)
#else
void DetailsWidget::enterEvent(QEnterEvent * event)
#endif
{
    QWidget::enterEvent(event);
    d->changeHoverState(true);
}

void DetailsWidget::leaveEvent(QEvent * event)
{
    QWidget::leaveEvent(event);
    d->changeHoverState(false);
}

void DetailsWidget::setSummaryText(const QString &text)
{
    d->m_summaryLabel->setText(text);
    d->updateControls();
}

QString DetailsWidget::summaryText() const
{
    return d->m_summaryLabel->text();
}

QString DetailsWidget::additionalSummaryText() const
{
    return d->m_additionalSummaryLabel->text();
}

void DetailsWidget::setAdditionalSummaryText(const QString &text)
{
    d->m_additionalSummaryLabel->setText(text);
    d->m_additionalSummaryLabel->setVisible(!text.isEmpty());
}

DetailsWidget::State DetailsWidget::state() const
{
    return d->m_state;
}

void DetailsWidget::setState(State state)
{
    if (state == d->m_state)
        return;
    d->m_state = state;
    d->updateControls();
    emit expanded(state == Expanded);
}

void DetailsWidget::setExpanded(bool expanded)
{
    setState(expanded ? Expanded : Collapsed);
}

QWidget *DetailsWidget::widget() const
{
    return d->m_widget;
}

//! Sets the 'detail' widget to be optionally shown when this widget is in its 'expanded' state
/*! \param widget the widget to be potentially displayed
 *  \param margin the margin applied to this widget. If omitted or negative
 *                  a default margin will be used instead. */
void DetailsWidget::setWidget(QWidget *widget, int margin)
{
    if (d->m_widget == widget)
        return;

    if (d->m_widget) {
        d->m_grid->removeWidget(d->m_widget);
        delete d->m_widget;
    }

    d->m_widget = widget;

    if (margin < 0)
        margin = MARGIN;

    if (d->m_widget) {
        d->m_widget->setContentsMargins(margin, margin, margin, margin);
        d->m_grid->addWidget(d->m_widget, 2, 0, 1, 5);
    }
    d->m_detailsButton->setCheckable(d->m_widget);
    d->updateControls();
}

void DetailsWidget::setToolWidget(QWidget *widget)
{
    if (d->m_toolWidget == widget)
        return;

    d->m_toolWidget = widget;

    if (!d->m_toolWidget)
        return;

    d->m_toolWidget->adjustSize();
    d->m_grid->addWidget(d->m_toolWidget, 0, 3, 1, 1, Qt::AlignRight);

    d->changeHoverState(d->m_hovered);

    setTabOrder(d->m_summaryCheckBox, d->m_toolWidget);
    setTabOrder(d->m_toolWidget, d->m_detailsButton);
}

QWidget *DetailsWidget::toolWidget() const
{
    return d->m_toolWidget;
}

bool DetailsWidget::eventFilter(QObject *obj, QEvent *event)
{
    if (obj != d->m_summaryLabel)
        return false;
    if (event->type() != QEvent::MouseButtonRelease)
        return false;
    QMouseEvent *mouseEvent = static_cast<QMouseEvent *>(event);
    if (mouseEvent->button() != Qt::LeftButton ||
        mouseEvent->buttons() != Qt::MouseButtons())
        return false;
    if (d->m_summaryCheckBox->isVisible() &&
        d->m_summaryCheckBox->isEnabled()) {
        d->m_summaryCheckBox->toggle();
        return true;
    }
    return false;
}

void DetailsWidget::summaryCheckBoxToggled(bool checked)
{
    if (d->m_detailsButtonEnabled &&
        d->m_detailsButton->isCheckable()) {
        int oldState = d->m_state;
        d->m_state = checked ? Expanded : Collapsed;
        d->updateControls();
        if (d->m_state != oldState)
            emit expanded(checked);
    } else {
        d->updateControls();
    }
}

void DetailsWidget::setDetailButtonVisible(bool b)
{
    d->m_detailsButtonVisible = b;
}

bool DetailsWidget::detailButtonVisible() const
{
    return d->m_detailsButtonVisible;
}

void DetailsWidget::setDetailButtonEnabled(bool b)
{
    d->m_detailsButtonEnabled = b;
    if (!b && d->m_state == Expanded) {
        d->m_state = Collapsed;
        d->updateControls();
        emit expanded(false);
    } else {
        d->updateControls();
    }
}

bool DetailsWidget::detailButtonEnabled() const
{
    return d->m_detailsButton->isEnabled();
}

void DetailsWidget::setSummaryToolTip(const QString& tip)
{
    if (d->m_summaryLabel)
        d->m_summaryLabel->setToolTip(tip);
    if (d->m_additionalSummaryLabel)
        d->m_additionalSummaryLabel->setToolTip(tip);
    if (d->m_toolWidget)
        d->m_toolWidget->setToolTip(tip);
}

void DetailsWidget::setDetailButtonText(const QString& text)
{
    setDetailButtonText(text, text);
}

void DetailsWidget::setDetailButtonText(const QString& expandText,
                                        const QString& collapseText)
{
    d->m_detailsButton->setDetailButtonText(expandText, collapseText);
}

void DetailsWidget::on_labelBackgroundClicked()
{
    if (d->m_detailsButton->isEnabled() && d->m_toggleDetailsOnSummaryBarClicks) {
        if (d->m_detailsButton->isCheckable())
            setExpanded(state() == Collapsed);
        else
            emit detailsClicked();
    }
}

//! Sets whether clicking on the summary line/bar toggles the visiblity of the 'details' portion of the widget
/*! The 'details' portion of the widget is normally toggled by clicking the 'details button', on
 *  the far right of the summary line (the top portion of the widget that is always visible).
 *  When this value is set to \c true clicking anywhere on the summary bar will have the same
 *  effect as clicking on the 'details button' (i.e. will toggle the visibility of the 'details'.
 *
 *  Set to \c true for user convienience when there is no need for the user to be clicking
 *  near the summary bar unless it is to toggle the details portion of the widget. Set to
 *  \c false if there are other interactive components on the summary bar (i.e. hyperlinks)
 *  - it is annoying for a misclicks to unintentionally toggle details. */
void DetailsWidget::setToggleDetailsOnSummaryBarClicks(bool b)
{
    d->m_toggleDetailsOnSummaryBarClicks = b;
}

//! Whether clicking on the summary line/bar toggles the visiblity of the 'details' portion of the widget
/*! \copydetails DetailsWidget::setToggleDetailsOnSummaryBarClicks */
bool DetailsWidget::toggleDetailsOnSummaryBarClicks() const
{
    return d->m_toggleDetailsOnSummaryBarClicks;
}


#include "detailswidget.moc"
