# SPDX-FileCopyrightText: Copyright (C) 2025 Linaro Limited (or its affiliates).
#
# -*- coding: iso-8859-1 -*-

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import gdb

def print_inferior_progspaces(progSpace):
    """
    Print all inferiors which match the given progspace.
    A match in this case is on the filename.
    """
    # Ensure we resolve any symbolic links as pointing to same executable
    filename = os.path.realpath(progSpace.executable_filename)
    inferiorArray = \
        [inferior.num for inferior in gdb.inferiors() \
         if inferior.progspace.executable_filename and \
            os.path.realpath(inferior.progspace.executable_filename) == filename]
    # We also hash here as is more easily parsable than any given filename
    data = {"file-hash": hash(filename), "inferiors": inferiorArray, "filename": filename}
    gdb.notify_mi("new-target", data)

def executable_changed_event(event):
    """
    Event for catching target changes.
    Print all inferiors matching the progspace filename, as well as a hash of the filename.
    """
    print_inferior_progspaces(event.progspace)

gdb.events.executable_changed.connect (executable_changed_event)
