# SPDX-FileCopyrightText: Copyright (C) 2025 Linaro Limited (or its affiliates). All rights reserved.
#
# -*- coding: iso-8859-1 -*-

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import gdb
import sys
import json

class ForgeMiTransaction(gdb.MICommand):
    """
    -forge-mi-transaction {"commands":[{"<command_1>", "<command_2>",...]}

    Takes in a list of commands to execute and returns a list of outputs.
    """
    def __init__(self):
        super (ForgeMiTransaction, self).__init__ ("-forge-mi-transaction")

    def invoke(self, args):
        """
        Run the command.
        """

        results_key = "results"
        results = []
        data = {results_key : results}
        if not args:
            return data

        commands = json.loads(" ".join(args))["commands"]
        for command in commands:
            try:
                # Is this an MI command? Need to use "execute_mi" instead.
                if command.startswith("-"):
                    command_list = command.split()
                    command_name = command_list[0]
                    arguments = [] if len(command_list) == 1 else command_list[1:]
                    results.append(gdb.execute_mi(command_name, *arguments))
                else:
                    results.append(gdb.execute(command, to_string=True))
            except gdb.error as error:
                # We don't want an error to stop execution of the other commands.
                # Print this error and append to results wrapped in <>.
                error_string = str(error)
                print(error_string, file=sys.stderr)
                results.append(f'<{error_string}>')

        return data

ForgeMiTransaction()
