# SPDX-FileCopyrightText: Copyright (C) 2025 Linaro Limited (or its affiliates). All rights reserved.
#
# -*- coding: iso-8859-1 -*-

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import gdb
import sys

class ForgeThreadInfoInferior(gdb.MICommand):
    """
    -forge-thread-info-inferior <inferior_num>

    A variation on the MI mode command "-thread-info" which will filter the results
    based on the given inferior number.
    """
    def __init__(self):
        super (ForgeThreadInfoInferior, self).__init__ ("-forge-thread-info-inferior")

    def invoke(self, args):
        """
        Run the command.
        """
        # Get the MI mode mode for -thread-info
        thread_info = gdb.execute_mi("-thread-info")
        if len(args) < 1:
            return thread_info

        # From arguments, get the inferior number which we are filtering.
        inferior_num = int(args[0])
        threads = thread_info["threads"]

        # Filter thread_info threads based on inferior number and return result.
        filtered_threads = list(thread_info_filter(threads, inferior_num))
        thread_info["threads"] = filtered_threads
        return thread_info

def thread_info_filter(threads, inferior):
    """
    Generator for filtering thread_info threads for a given inferior.
    """
    inferior_gids = list(inferrior_threads_filter(inferior))
    select = lambda thread: int(thread["id"]) in inferior_gids
    for thread in filter(select, threads):
        yield thread

def inferrior_threads_filter(inferior_num):
    """
    Generator for all thread GID's of a given inferior.
    """
    select = lambda inferior: inferior.num == inferior_num
    for inferior in filter(select, gdb.inferiors()):
        for thread in inferior.threads():
            yield thread.global_num

ForgeThreadInfoInferior()
