# Frame Filter for thread starter functions.
#
# SPDX-FileCopyrightText: Copyright (C) 2023-2025 Linaro Limited (or its affiliates). All rights reserved. Copyright (C) 2023 Arm Limited and/or its affiliates.
#
# -*- coding: iso-8859-1 -*-

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import gdb
from gdb.FrameDecorator import FrameDecorator

import collections

class ThreadStartFilterDecorator(FrameDecorator):
    """
    Decorator for non-main thread filter. Make sure that frames
    with no function name are given a name of "??" for consistency
    """
    def __init__(self, fobj):
        super(ThreadStartFilterDecorator, self).__init__(fobj)

    def function(self):
        frame = self.inferior_frame()
        name = frame.name()

        # Keep as ?? if there is no name for consistency
        if not name:
            return "??"

        return str(name)

class ThreadStartFilter():
    """
    Filter the non-main thread start functions from the stacks:
    1) Cosmetic: It matches the hiding of the libc calls on the main thread.
    2) Provides a common base for the stack alignment feature in the CTC window.
    """
    def __init__(self):
        self.name = "ThreadStartFilter"
        self.priority = 1000
        self.enabled = True
        gdb.frame_filters[self.name] = self

    def filter(self, frame_iter):
        """
        For optimal performance we are returning a generator here meaning
        we are only iterating through the frame once during this filter.
        This is the recommended approach by GDB.
        """
        # Frames we want to filter out
        threads = ["clone", "__clone2", "clone3",
                    "_pthread_body", "start_thread", "thread_start"]

        # The rejected frames (might be inserted back if not at the end)
        rejected_frames = collections.deque()

        # Variable for checking if we yielded any frames at all
        empty_stack = True

        # Itterate through all frames, only removing frames if they
        # are the first frames in the stack
        for frame in frame_iter:
            if frame.inferior_frame().name() in threads:
                rejected_frames.append(frame)
            else:
                # Not at the end, go through rejected frames first!
                while len(rejected_frames) > 0:
                    yield ThreadStartFilterDecorator(rejected_frames.popleft())
                yield ThreadStartFilterDecorator(frame)
                empty_stack = False

        # If nothing was yielded at all, better to return the thread start
        # stack then nothing at all!
        if empty_stack:
            while len(rejected_frames) > 0:
                yield ThreadStartFilterDecorator(rejected_frames.popleft())


ThreadStartFilter()
