# SPDX-FileCopyrightText: Copyright (C) 2023-2025 Linaro Limited (or its affiliates). All rights reserved. Copyright (C) 2023 Arm Limited and/or its affiliates.
#
# -*- coding: iso-8859-1 -*-

# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from __future__ import print_function
import os
import gdb

def log_info(*args, **kwargs):
    log = False
    if log:
        print(*args, **kwargs)

def new_objfile_event (event):
    """
    Load auto-load code that is shipped with Forge. This is needed because GDB is set up to load
    code resident next to libraries or from platform specific files in the data directory. Where
    as Forge ships it's own auto-load code in a flat directory structure.
    """
    objfile = event.new_objfile
    progspace = objfile.progspace
    libstdcpp_basename = "libstdc++.so.6"
    autoload_suffix = "-gdb.py"
    auto_dir = os.path.join(os.path.dirname (__file__), '../../../auto-load')

    if objfile.filename.endswith(libstdcpp_basename):
        progspace.found_lib_cxx = True

    autoload_file = os.path.basename(objfile.filename) + autoload_suffix
    autoload_candidate = os.path.join(auto_dir, autoload_file)
    if os.path.exists(autoload_candidate):
        log_info("Detected auto-load for", objfile.filename, "at", autoload_candidate, "Loading.")
        load_auto_load(autoload_candidate)

def gcc_personality_not_found(progspace):
    """
    Return true if the gcc_personality has not yet been found for the given program space.
    """
    return not hasattr(progspace, 'found_lib_cxx') or not progspace.found_lib_cxx

def load_auto_load(filename):
    """
    Execute the auto load code in the current environment.
    """
    with open(filename, 'rb') as autoload:
        exec(autoload.read())

def stop_event (event):
    """
    If the GCC personality can be found at the first stop, then load the pretty printers.
    This accounts for the case where libstdc++ has ben statically linked into the user's
    application.
    """
    gcc_personality = "__gxx_personality_v0"

    try:
        output = gdb.parse_and_eval (gcc_personality)
    except gdb.error:
        pass # Symbol not found. Carry on.

    if output:
        from libstdcxx.v6 import register_libstdcxx_printers
        register_libstdcxx_printers (None)

    gdb.events.stop.disconnect (stop_event)

gdb.events.new_objfile.connect (new_objfile_event)
gdb.events.stop.connect (stop_event)
