#!/usr/bin/env python
'''
This Python script demonstrates MAP's Python profiling capabilities.
To run the demo, from the examples directory compile the C libraries with:
     $ make -f python-profiling.makefile
then run
     $ ../bin/map --start python python-profiling.py --seconds 15
or
     $ ../bin/map --start python python-profiling.py --index 30
'''
import os
import ctypes
import argparse
import time

example_dir = os.path.dirname(os.path.realpath(__file__))

fibonacci_lib = ctypes.CDLL(os.path.join(example_dir, 'libfibonacci.so'))
fibonacci_lib.compute_fibonacci.argtypes = [ctypes.c_ulonglong]
fibonacci_lib.compute_fibonacci.restype = ctypes.c_ulonglong

logistic_map_lib = ctypes.CDLL(os.path.join(example_dir, 'liblogisticmap.so'))
logistic_map_lib.compute_logistic_map.argtypes = [ctypes.c_double, ctypes.c_double, ctypes.c_uint]
logistic_map_lib.compute_logistic_map.restype = ctypes.c_ulonglong


def fibonacci_c(index):
    """
    Compute the Fibonacci number at the specified index using C.
    """
    return fibonacci_lib.compute_fibonacci(index)

def fibonacci_python(index):
    """
    Compute the Fibonacci number at the specified index using Python.
    """
    return index if index in [0, 1] else fibonacci_python(index-1) + fibonacci_python(index-2)

def logistic_map_c(r, x, time_seconds):
    """
    Performa a logistic map calculation for the specified number of
    seconds using C.
    """
    return logistic_map_lib.compute_logistic_map(r, x, time_seconds)

def logistic_map_python(r, x, time_seconds):
    """
    Performa a logistic map calculation for the specified number of
    seconds using Python.
    """
    start_time = time.time()
    iterations = 0
    while time.time() - start_time < time_seconds:
        for p in range(0, 30):
            x = r*x*(1.0-x)
            iterations += 1
    return iterations

def main():
    parser = argparse.ArgumentParser(
        description="Compare C and Python computation by either perfoming logistic map "
                    "calculations for a fixed duration or by computing a Fibonacci number.")
    group = parser.add_mutually_exclusive_group()
    group.add_argument("--index", dest="index", type=int,
                       help="index in the Fibonacci sequence")
    group.add_argument("--seconds", dest="seconds", type=int, default=15,
                       help="force C and Python computation for this duration each")
    options = parser.parse_args()

    if not options.index is None:
        print("Calculating the Fibonacci number at index %s in both C and Python\n"
              "(use --index to specify another index)\n" % options.index)
        print("  %s (computed in C)" % fibonacci_c(options.index))
        print("  %s (computed in Python)" % fibonacci_python(options.index))
    else:
        print("Computing for %s seconds each in both C and Python\n"
              "(use --seconds to specify another duration)\n" % options.seconds)
        print("  %s iterations computed in C" % logistic_map_c(3.8, 0.5, options.seconds))
        print("  %s iterations computed in Python" % logistic_map_python(3.8, 0.5, options.seconds))

if __name__ == "__main__":
    main()
