/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtCore module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

/****************************************************************************
**
** Copyright (C) 2023-2025 Linaro Limited (or its affiliates). All rights reserved.
** Copyright (C) 2012-2023 Arm Limited (or its affiliates).
**
****************************************************************************/
#include "qpipewriter_p.h"
#include <QSocketNotifier>
#include <sys/ioctl.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>

Q_PIPE qt_open_write_pipe(const char* pathname)
{
    return ::open(pathname, O_WRONLY | O_NONBLOCK);
}

qint64 qt_native_write(int fd, const char *data, qint64 len)
{
    qint64 ret = 0;
    do {
        ret = ::write(fd, data, len);
    } while (ret == -1 && errno == EINTR);
    return ret;
}

void qt_ignore_sigpipe()
{
    // Set to ignore SIGPIPE once only.
    static QBasicAtomicInt atom = Q_BASIC_ATOMIC_INITIALIZER(0);
    if (atom.testAndSetRelaxed(0, 1)) {
        struct sigaction noaction;
        memset(&noaction, 0, sizeof(noaction));
        noaction.sa_handler = SIG_IGN;
        sigaction(SIGPIPE, &noaction, 0);
    }
}

void QPipeWriterPrivate::closePipe()
{
    if (pipe != INVALID_Q_PIPE)
    {
        ::close(pipe);
        pipe = INVALID_Q_PIPE;
    }
}

void QPipeWriterPrivate::createNotifier()
{
    Q_Q(QPipeWriter);
    delete notifier;
    notifier = new QSocketNotifier(pipe, QSocketNotifier::Write);
    notifier->setEnabled(true);
    QObject::connect(notifier, SIGNAL(activated(QSocketDescriptor, QSocketNotifier::Type)), q, SLOT(_q_canWrite()));
}

qint64 QPipeWriterPrivate::bytesAvailable() const
{
    // According to ioctl_list man page FIONREAD should
    // use a int* as its data type. 
    int nbytes = 0;
    qint64 available = -1;
    if (::ioctl(pipe, FIONREAD, &nbytes) >= 0)
        available = static_cast<qint64>(nbytes);
#if defined (QPIPEWRITER_DEBUG)
    qDebug("QPipeWriterPrivate::bytesAvailable() == %lld", available);
#endif
    return available;
}
