/**
 * Copyright (C) 2023-2025 Linaro Limited (or its affiliates). All rights reserved.
 * Copyright (C) 2012-2023 Arm Limited (or its affiliates).
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */

//! \file
//! **PathString** definition.

#ifndef PATHSTRING_H
#define PATHSTRING_H

// +explicit-local-file: All QFileInfo and QDir usage operates on strings only.
#include "lgpl_core_export.h"
#include <QString>
#include <QStringList>
#include "internstringcache.h"

//! Wrapper for QString for manipulating file names.
/*!
 * PathString wraps QFileInfo to provide a QString based interface that does not
 * access the filesystem.  This class emphasises that the path in question is
 * free from the context of a particular host machine.  Both Unix- and
 * Windows-style paths are accepted.
*/
class LGPL_CORE_EXPORT PathString
{
public:
    PathString() {} //!< Construct an empty PathString.
    explicit PathString(const QString &string);
    explicit PathString(const QString &parent, const QString& filename);
    QString fileName() const;
    QString path() const;
    QString filePath() const;
    QString filePath(const QString &fileName) const;
    QString suffix() const;
    QString completeBaseName() const;
    bool isRelative() const;
    bool isAbsolute() const;
    bool isRoot() const;
    bool isEmpty() const;
    QStringList splitPath() const;
    static QString joinPath(const QStringList& parts, const int pos=0, const int length=-1);
    static QString cleanPath(const QString &path);
    static bool fileNamesEqual (QString const &f1, QString const &f2);

private:
    QString mFileName;
};

//! Construct a PathString from a given file name.
inline
PathString::PathString(const QString &fileName) :
    mFileName(InternStringCache::intern(fileName))
{
}

//! Construct a PathString from a given file name.
/*!
 * \param  parent    A directory.
 * \param  fileName  A file inside the directory.
 */
inline
PathString::PathString(const QString &parent, const QString& fileName) :
    mFileName(InternStringCache::intern(PathString(parent).filePath(fileName)))
{
}

//! Return the full file name, directories included.
inline
QString PathString::filePath() const
{
    return mFileName;
}

//! Return `true` if no file name is stored, `false` otherwise.
inline
bool
PathString::isEmpty() const
{
    return mFileName.isEmpty();
}

//! Return PathString::fileNamesEqual().
inline
bool
operator== (PathString const &f1, PathString const &f2)
{
	return PathString::fileNamesEqual (f1.filePath (), f2.filePath ());
}

//! \overload bool operator== (PathString const &f1, PathString const &f2)
inline
bool
operator== (PathString const &f1, QString const &f2)
{
	return f1 == PathString (f2);
}

//! \overload bool operator== (PathString const &f1, PathString const &f2)
inline
bool
operator== (QString const &f1, PathString const &f2)
{
	return PathString (f1) == f2;
}

#endif // PATHSTRING_H
