/**************************************************************************
**
** This file is part of Qt Creator
**
** Copyright (c) 2012 Nokia Corporation and/or its subsidiary(-ies).
**
** Contact: Nokia Corporation (qt-info@nokia.com)
**
**
** GNU Lesser General Public License Usage
**
** This file may be used under the terms of the GNU Lesser General Public
** License version 2.1 as published by the Free Software Foundation and
** appearing in the file LICENSE.LGPL included in the packaging of this file.
** Please review the following information to ensure the GNU Lesser General
** Public License version 2.1 requirements will be met:
** http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** Other Usage
**
** Alternatively, this file may be used in accordance with the terms and
** conditions contained in a signed written agreement between you and Nokia.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
**
**************************************************************************/

#include "fakevimactions.h"
#include "fakevimhandler.h"

#include "../utils/savedaction.h"

// Please do not add any direct dependencies to other Qt Creator code  here.
// Instead emit signals and let the FakeVimPlugin channel the information to
// Qt Creator. The idea is to keep this file here in a "clean" state that
// allows easy reuse with any QTextEdit or QPlainTextEdit derived class.


#include "../utils/qtcassert.h"

#include <QDebug>
#include <QObject>
#include <QCoreApplication>

using namespace Utils;

///////////////////////////////////////////////////////////////////////
//
// FakeVimSettings
//
///////////////////////////////////////////////////////////////////////

namespace TextEditor {
namespace FakeVim {

FakeVimSettings::FakeVimSettings()
{}

FakeVimSettings::~FakeVimSettings()
{
    qDeleteAll(m_items);
}

void FakeVimSettings::insertItem(int code, SavedAction *item,
    const QString &longName, const QString &shortName)
{
    QTC_ASSERT(!m_items.contains(code), qDebug() << code << item->toString(); return);
    m_items[code] = item;
    if (!longName.isEmpty()) {
        m_nameToCode[longName] = code;
        m_codeToName[code] = longName;
    }
    if (!shortName.isEmpty()) {
        m_nameToCode[shortName] = code;
    }
}

void FakeVimSettings::readSettings(QSettings *settings)
{
    foreach (SavedAction *item, m_items)
        item->readSettings(settings);
}

void FakeVimSettings::writeSettings(QSettings *settings)
{
    foreach (SavedAction *item, m_items)
        item->writeSettings(settings);
}

SavedAction *FakeVimSettings::item(int code)
{
    QTC_ASSERT(m_items.value(code, 0), qDebug() << "CODE: " << code; return 0);
    return m_items.value(code, 0);
}

SavedAction *FakeVimSettings::item(const QString &name)
{
    return m_items.value(m_nameToCode.value(name, -1), 0);
}

QString FakeVimSettings::trySetValue(const QString &name, const QString &value)
{
    int code = m_nameToCode.value(name, -1);
    if (code == -1)
        return FakeVimHandler::tr("Unknown option: %1").arg(name);
    if (code == ConfigTabStop || code == ConfigShiftWidth) {
        if (value.toInt() <= 0)
            return FakeVimHandler::tr("Argument must be positive: %1=%2")
                    .arg(name).arg(value);
    }
    SavedAction *act = item(code);
    if (!act)
        return FakeVimHandler::tr("Unknown option: %1").arg(name);
    act->setValue(value);
    return QString();
}

FakeVimSettings *theFakeVimSettings()
{
    static FakeVimSettings *instance = 0;
    if (instance)
        return instance;

    instance = new FakeVimSettings;

    //typedef QLatin1String _;
    SavedAction *item = 0;

    const QString group = QLatin1String("FakeVim");
    item = new SavedAction(instance);
    item->setText(QCoreApplication::translate("FakeVim::Internal",
        "Use Vim-style Editing"));
    item->setSettingsKey(group, QLatin1String("UseFakeVim"));
    item->setCheckable(true);
    item->setValue(false);
    instance->insertItem(ConfigUseFakeVim, item);

    item = new SavedAction(instance);
    item->setText(QCoreApplication::translate("FakeVim::Internal",
        "Read .vimrc"));
    item->setSettingsKey(group, QLatin1String("ReadVimRc"));
    item->setCheckable(true);
    item->setValue(false);
    instance->insertItem(ConfigReadVimRc, item);

    item = new SavedAction(instance);
    item->setValue(true);
    item->setDefaultValue(true);
    item->setSettingsKey(group, QLatin1String("StartOfLine"));
    item->setCheckable(true);
    instance->insertItem(ConfigStartOfLine, item, QLatin1String("startofline"), QLatin1String("sol"));

    item = new SavedAction(instance);
    item->setDefaultValue(8);
    item->setSettingsKey(group, QLatin1String("TabStop"));
    instance->insertItem(ConfigTabStop, item, QLatin1String("tabstop"), QLatin1String("ts"));

    item = new SavedAction(instance);
    item->setDefaultValue(false);
    item->setValue(false);
    item->setSettingsKey(group, QLatin1String("SmartTab"));
    instance->insertItem(ConfigSmartTab, item, QLatin1String("smarttab"), QLatin1String("sta"));

    item = new SavedAction(instance);
    item->setDefaultValue(true);
    item->setValue(true);
    item->setSettingsKey(group, QLatin1String("HlSearch"));
    item->setCheckable(true);
    instance->insertItem(ConfigHlSearch, item, QLatin1String("hlsearch"), QLatin1String("hls"));

    item = new SavedAction(instance);
    item->setDefaultValue(8);
    item->setSettingsKey(group, QLatin1String("ShiftWidth"));
    instance->insertItem(ConfigShiftWidth, item, QLatin1String("shiftwidth"), QLatin1String("sw"));

    item = new SavedAction(instance);
    item->setDefaultValue(false);
    item->setValue(false);
    item->setSettingsKey(group, QLatin1String("ExpandTab"));
    item->setCheckable(true);
    instance->insertItem(ConfigExpandTab, item, QLatin1String("expandtab"), QLatin1String("et"));

    item = new SavedAction(instance);
    item->setDefaultValue(false);
    item->setValue(false);
    item->setSettingsKey(group, QLatin1String("AutoIndent"));
    item->setValue(false);
    item->setCheckable(true);
    instance->insertItem(ConfigAutoIndent, item, QLatin1String("autoindent"), QLatin1String("ai"));

    item = new SavedAction(instance);
    item->setDefaultValue(false);
    item->setValue(false);
    item->setSettingsKey(group, QLatin1String("SmartIndent"));
    item->setValue(false);
    item->setCheckable(true);
    instance->insertItem(ConfigSmartIndent, item, QLatin1String("smartindent"), QLatin1String("si"));

    item = new SavedAction(instance);
    item->setDefaultValue(true);
    item->setValue(true);
    item->setSettingsKey(group, QLatin1String("IncSearch"));
    item->setCheckable(true);
    instance->insertItem(ConfigIncSearch, item, QLatin1String("incsearch"), QLatin1String("is"));

    item = new SavedAction(instance);
    item->setDefaultValue(false);
    item->setValue(false);
    item->setSettingsKey(group, QLatin1String("UseCoreSearch")); item->setCheckable(true);
    instance->insertItem(ConfigUseCoreSearch, item,
        QLatin1String("usecoresearch"), QLatin1String("ucs"));

    item = new SavedAction(instance);
    item->setDefaultValue(QLatin1String("indent,eol,start"));
    item->setSettingsKey(group, QLatin1String("Backspace"));
    instance->insertItem(ConfigBackspace, item, QLatin1String("backspace"), QLatin1String("bs"));

    item = new SavedAction(instance);
    item->setDefaultValue(QLatin1String("@,48-57,_,192-255,a-z,A-Z"));
    item->setSettingsKey(group, QLatin1String("IsKeyword"));
    instance->insertItem(ConfigIsKeyword, item, QLatin1String("iskeyword"), QLatin1String("isk"));

    // Invented here.
    item = new SavedAction(instance);
    item->setDefaultValue(false);
    item->setValue(false);
    item->setSettingsKey(group, QLatin1String("ShowMarks"));
    item->setCheckable(true);
    instance->insertItem(ConfigShowMarks, item, QLatin1String("showmarks"), QLatin1String("sm"));

    item = new SavedAction(instance);
    item->setDefaultValue(false);
    item->setValue(false);
    item->setSettingsKey(group, QLatin1String("PassControlKey"));
    item->setCheckable(true);
    instance->insertItem(ConfigPassControlKey, item, QLatin1String("passcontrolkey"), QLatin1String("pck"));

    return instance;
}

SavedAction *theFakeVimSetting(int code)
{
    return theFakeVimSettings()->item(code);
}

} // namespace FakeVim
} // namespace TextEditor
