/**
 * Copyright (C) 2023-2025 Linaro Limited (or its affiliates). All rights reserved.
 * Copyright (C) 2012-2023 Arm Limited (or its affiliates).
 * All rights reserved.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 */

#ifndef SECTIONMAP_H
#define SECTIONMAP_H

#include "lgpl_gui_export.h"

/* SectionMap maps logical sections to physical (display) sections and vice
   versa. It is used by MDATable. The code is modified from the red-black
   tree implementation used by QMap. Each node keeps a running total of its
   left and right branches. This allows us to map logical to physical sections
   and vice versa in O(log(n)) time (as opposed to QTable's original O(n)
   algorithm). */
// See http://e98cuenc.free.fr/wordprocessor/piecetable.html for a description
// of a similar data structure.

struct SectionMapNode
{
    enum Color { Red, Black };

    SectionMapNode( const long long _key, const long long _count = 1 )
    {
        key = _key;
        leftCount = 0;
        rightCount = 0;
        count = _count;
    }

    SectionMapNode * left;
    SectionMapNode * right;
    SectionMapNode * parent;

    Color color;

    long long count;
    long long leftCount;
    long long rightCount;
    long long key;
};


// ### 4.0: rename to something without Private in it. Not really internal.
class SectionMapPrivateBase
{
public:
    /**
     * Typedefs
     */
    typedef SectionMapNode Node;
    typedef SectionMapNode *NodePtr;

    SectionMapPrivateBase()
    {
        root = 0;
    }

    /**
     * Implementations of basic tree algorithms
     */
    void rotateLeft( NodePtr x );
    void rotateRight( NodePtr x );
    void rebalance( NodePtr x );
    NodePtr removeAndRebalance( NodePtr z );

    /**
     * Variables
     */
    NodePtr root;
};


class SectionMapPrivate : protected SectionMapPrivateBase
{
public:
    /**
     * Functions
     */
    ~SectionMapPrivate()
    {
        clear();
    }

    void clear();
    void clear( NodePtr p );


#ifdef SECTIONMAP_DEBUG
    void inorder( NodePtr x = 0, int level = 0 );
    void treeorder( NodePtr x = 0, int level = 0 );
    void fixup( NodePtr x = 0);

    long long hiddenCount( NodePtr x );
#endif

protected:
    /**
     * Helpers
     */
    long long key( NodePtr b ) const { return ((NodePtr)b)->key; }

    void fixParents( NodePtr x, long long delta );
    NodePtr find( const long long k, bool exact = true ) const;

    NodePtr insertSingle( const long long k, const long long count = 1 );
    NodePtr insert( NodePtr y, const long long k, const long long count = 1 );
};

class SectionMap : public SectionMapPrivate
{
public:
    void hideSection(long long s);
    void showSection(long long s);
    void hideSections(long long s, long long e);
    void showSections(long long s, long long e);
    bool isSectionHidden(long long s) const;
    bool isSectionHidden(long long s, long long e) const;
    long long sectionPos(long long s) const;
    long long sectionAt(long long s) const;
};

#endif // SECTIONMAP_H
