/****************************************************************************
**
** Copyright (C) 2009 Nokia Corporation and/or its subsidiary(-ies).
** Contact: Qt Software Information (qt-info@nokia.com)
**
** This file is part of the QtCore module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain
** additional rights. These rights are described in the Nokia Qt LGPL
** Exception version 1.0, included in the file LGPL_EXCEPTION.txt in this
** package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you are unsure which license is appropriate for your use, please
** contact the sales department at qt-sales@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

/****************************************************************************
**
** Copyright (C) 2023-2025 Linaro Limited (or its affiliates). All rights reserved.
** Copyright (C) 2012-2023 Arm Limited (or its affiliates).
**
****************************************************************************/
#include "qpipereader_p.h"
#include <QSocketNotifier>
#include <sys/ioctl.h>
#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <unistd.h>

Q_PIPE qt_open_read_pipe(const char* pathname)
{
return ::open(pathname, O_RDONLY | O_NONBLOCK);
}

qint64 qt_native_read(int fd, char *data, qint64 maxlen)
{
    qint64 ret = 0;
    do {
        ret = ::read(fd, data, maxlen);
    } while (ret == -1 && errno == EINTR);
    return ret;
}

void QPipeReaderPrivate::closePipe()
{
    if (pipe != INVALID_Q_PIPE)
    {
        ::close(pipe);
        pipe = INVALID_Q_PIPE;
    }
}

void QPipeReaderPrivate::createNotifier()
{
    Q_Q(QPipeReader);
    delete notifier;
    notifier = new QSocketNotifier(pipe, QSocketNotifier::Read);
    notifier->setEnabled(true);
    QObject::connect(notifier, SIGNAL(activated(QSocketDescriptor, QSocketNotifier::Type)), q, SLOT(_q_canRead()));
}

qint64 QPipeReaderPrivate::bytesAvailable() const
{
    // According to ioctl_list man page FIONREAD should
    // use a int* as its data type. 
    int nbytes = 0;
    qint64 available = -1;
    if (::ioctl(pipe, FIONREAD, &nbytes) >= 0)
        available = static_cast<qint64>(nbytes);

#if defined (QPIPEREADER_DEBUG)
    qDebug("QPipeReaderPrivate::bytesAvailable() == %lld", available);
#endif
    return available;
}

/*! \internal
*/
bool QPipeReaderPrivate::_q_canRead(bool drain)
{
    Q_Q(QPipeReader);
    qint64 available;
    if (blocking) {
        available = bytesAvailable();
        if (available == 0) {
            if (drain) {
                // 0 bytes available is not a problem if called from drain.
                return false;
            } else {
                // #16217: QPipeReader - bogus end of file
                // Sometimes the socket notifier activates just after drain. I suspect select/poll/epoll signals the socket is
                // ready, then some other queued slot is invoked which calls drain, then the socket notifier activates and
                // this method is called. By this point there is nothing left in the pipe since drain read it all so
                // bytesAvailable will return 0. But this does *not* signal the end of file. So we need to double check here.
                struct pollfd pollfd;
                pollfd.events = POLLIN;
                pollfd.fd = pipe;
                pollfd.revents = 0;

                int ret = poll(&pollfd, 1, 0);
                if (ret < 0) {
                    eof = true;
                    if (notifier)
                        notifier->setEnabled(false);
#if defined QPIPEREADER_DEBUG
                    qDebug("QPipeReaderPrivate::canRead(), end of file");
#endif
                    emit q->eof();
                    emit q->readyRead();
                    return false;
                } else if (ret > 0) {
                    // ALL-2094: Sometimes ioctl reports 0 bytes available
                    // but poll disagrees. Try to read - we should get a byte
                    // or an EOF.
                    available = 1;
                } else {
                    // Not really the end of file.
                    return false;
                }
            }
        } else if (available == -1) {
            // ioctl(FIONREAD, ...) failed (e.g. with ENOTTY)
            available = 1;
        }
    } else {
        available = 4096;
    }

    qint64 totalReadBytes = 0;
    do {
        char *ptr = buffer.reserve(available);
        qint64 readBytes = read(ptr, available);
        if (readBytes == -1) {
            if (errno == EAGAIN || errno == EWOULDBLOCK) {
                buffer.chop(available);
                break;
            } else if (errno == EBADF) {
                // Treat EBADF as EOF.
                readBytes = 0;
            } else {
                buffer.chop(available);
                q->setErrorString(QPipeReader::tr("read error"));
                emit q->readError();
#if defined QPIPEREADER_DEBUG
                qDebug("QPipeReaderPrivate::canRead(), failed to read from the pipe");
#endif
                return false;
            }
        }
#if defined QPIPEREADER_DEBUG
        qDebug("QPipeReaderPrivate::canRead, read %d bytes from the pipe",
                int(readBytes));
#endif

        buffer.chop(available - readBytes);

        if (readBytes == 0) {
            eof = true;
            if (notifier)
                notifier->setEnabled(false);
#if defined QPIPEREADER_DEBUG
            qDebug("QPipeReaderPrivate::canRead(), end of file");
#endif
            emit q->eof();
            break;
        } else if (logFile) {
            logFile->write(ptr, readBytes);
            logFile->flush();
        }
        totalReadBytes += readBytes;
    } while (!blocking);
    if (!drain)
        emit q->readyRead();
    return totalReadBytes > 0;
}
