/**************************************************************************
**
** This file is part of Qt Creator
**
** Copyright (c) 2011 Nokia Corporation and/or its subsidiary(-ies).
**
** Contact: Nokia Corporation (info@qt.nokia.com)
**
**
** GNU Lesser General Public License Usage
**
** This file may be used under the terms of the GNU Lesser General Public
** License version 2.1 as published by the Free Software Foundation and
** appearing in the file LICENSE.LGPL included in the packaging of this file.
** Please review the following information to ensure the GNU Lesser General
** Public License version 2.1 requirements will be met:
** http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights. These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** Other Usage
**
** Alternatively, this file may be used in accordance with the terms and
** conditions contained in a signed written agreement between you and Nokia.
**
** If you have questions regarding the use of this file, please contact
** Nokia at info@qt.nokia.com.
**
**************************************************************************/

#include "detailsbutton.h"

#include <QPaintEvent>
#include <QPainter>
#include <QStyleOption>
#include <QGraphicsOpacityEffect>
#include <QPaintEngine>
#include <QColormap>
#include <QPropertyAnimation>

FadingPanel::FadingPanel(QWidget *parent) :
    QWidget(parent),
    m_opacityEffect(new QGraphicsOpacityEffect)
{
    m_opacityEffect->setOpacity(0);
    setGraphicsEffect(m_opacityEffect);

    // Workaround for issue with QGraphicsEffect. GraphicsEffect
    // currently clears with Window color. Remove if flickering
    // no longer occurs on fade-in
    QPalette pal;
    pal.setBrush(QPalette::All, QPalette::Window, Qt::transparent);
    setPalette(pal);

    m_useTransparency = QColormap::instance().depth() >= 16;
}

void FadingPanel::setOpacity(qreal value)
{
    m_opacityEffect->setOpacity(value);
}

void FadingPanel::fadeTo(float value)
{
    if (m_useTransparency) {
        QPropertyAnimation *animation = new QPropertyAnimation(m_opacityEffect, "opacity");
        animation->setDuration(200);
        animation->setEndValue(value);
        animation->start(QAbstractAnimation::DeleteWhenStopped);
    } else {
        setOpacity(value);
    }
}

DetailsButton::DetailsButton(QWidget *parent) : QAbstractButton(parent), m_fader(0)
{
    setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Minimum);
    setText(tr("Details..."));
    m_useTransparency = QColormap::instance().depth() >= 16;
    m_useGradient = QColormap::instance().depth() >= 16;
}

QSize DetailsButton::sizeHint() const
{
    // TODO: Adjust this when icons become available!
    int w;
    if (m_collapseText.isEmpty() && m_expandText.isEmpty()) {
        if (isCheckable())
            w = fontMetrics().horizontalAdvance(tr("Details"));
        else
            w = fontMetrics().horizontalAdvance(tr("Details..."));
    } else {
        w = qMax(fontMetrics().horizontalAdvance(m_collapseText),
                 fontMetrics().horizontalAdvance(m_expandText));
    }
    w += 32;
     
#ifdef Q_OS_MACOS
    return QSize(w, 34);
#else
    return QSize(w, 22);
#endif
}

bool DetailsButton::event(QEvent *e)
{
    switch(e->type()) {
    case QEvent::Enter:
        if (m_useTransparency)
        {
            QPropertyAnimation *animation = new QPropertyAnimation(this, "fader");
            animation->setDuration(200);
            animation->setEndValue(1.0);
            animation->start(QAbstractAnimation::DeleteWhenStopped);
        }
        else
            setFader(1.0);
        break;
    case QEvent::Leave:
        if (m_useTransparency)
        {
            QPropertyAnimation *animation = new QPropertyAnimation(this, "fader");
            animation->setDuration(200);
            animation->setEndValue(0.0);
            animation->start(QAbstractAnimation::DeleteWhenStopped);
        }
        else
            setFader(0.0);
        break;
    default:
        return QAbstractButton::event(e);
    }
    return false;
}

void DetailsButton::paintEvent(QPaintEvent *e)
{
    QWidget::paintEvent(e);

    QPainter p(this);
#ifndef Q_OS_MACOS
    // draw hover animation
    if(m_useTransparency && !isDown() && m_fader>0)
        p.fillRect(rect().adjusted(1, 1, -2, -2), QColor(255, 255, 255, int(m_fader*180)));
#endif
    QPixmap pixmap(cacheRendering(contentsRect().size(), false, isChecked(), m_useTransparency ? false : isDown(), m_useTransparency ? false : (!isDown() && m_fader>0), isEnabled()));
    p.drawPixmap(contentsRect(), pixmap);
    if (isDown() && m_useTransparency) {
        p.setPen(Qt::NoPen);
        p.setBrush(QColor(0, 0, 0, 20));
        p.drawRoundedRect(rect().adjusted(1, 1, -1, -1), 1, 1);
    }
}

QPixmap DetailsButton::cacheRendering(const QSize &size, bool checkable, bool checked, bool down, bool hover, bool enabled)
{
    PixmapCacheKey key(size, checkable, checked, down, hover, enabled);
    QPixmap pixmap(m_cachedPixmaps.value(key));
    if (!pixmap.isNull())
        return pixmap;

    QLinearGradient lg;
    lg.setCoordinateMode(QGradient::ObjectBoundingMode);
    lg.setFinalStop(0, 1);

    pixmap = QPixmap(size);
    QColor bg;
    if(m_useTransparency) {
        pixmap.fill(Qt::transparent);
    } else {
        bg = palette().color(backgroundRole());
        pixmap.fill(bg);
    }

    QPainter p(&pixmap);
    if (hover) {
        QColor hoverColour(mix(QColor(255, 255, 255, int(m_fader*180)), bg));
        p.fillRect(rect().adjusted(1, 1, -2, -2), hoverColour);
        bg = hoverColour;
    }

    if (down) {
        QColor downColour(mix(QColor(0, 0, 0, 20), bg));
        p.setPen(Qt::NoPen);
        p.setBrush(downColour);
        p.drawRoundedRect(rect().adjusted(1, 1, -1, -1), 1, 1);
        bg = downColour;
    }

    p.setRenderHint(QPainter::Antialiasing, true);
    p.translate(0.5, 0.5);
    p.setPen(Qt::NoPen);

    if (m_useGradient) {
        if(!checked) {
            if (m_useTransparency) {
                lg.setColorAt(0, QColor(0, 0, 0, 10));
                lg.setColorAt(1, QColor(0, 0, 0, 16));
            } else {
                lg.setColorAt(0, mix(QColor(0, 0, 0, 10), bg));
                lg.setColorAt(1, mix(QColor(0, 0, 0, 16), bg));
            }
        } else {
            if (m_useTransparency) {
                lg.setColorAt(0, QColor(255, 255, 255, 0));
                lg.setColorAt(1, QColor(255, 255, 255, 50));
            } else {
                lg.setColorAt(0, mix(QColor(255, 255, 255, 0), bg));
                lg.setColorAt(1, mix(QColor(255, 255, 255, 50), bg));
            }
        }
        p.setBrush(lg);
    } else {
        p.setBrush(Qt::NoBrush);
    }

    if (m_useTransparency)
        p.setPen(QColor(255,255,255,140));
    else
        p.setPen(mix(QColor(255,255,255,140), bg));
    p.drawRoundedRect(1, 1, size.width()-3, size.height()-3, 1, 1);
    if (m_useTransparency)
        p.setPen(QPen(QColor(0, 0, 0, 40)));
    else
        p.setPen(QPen(mix(QColor(0, 0, 0, 40), bg)));
    p.drawLine(0, 1, 0, size.height() - 2);
    if(checked)
        p.drawLine(1, size.height() - 1, size.width() - 1, size.height() - 1);
    p.setPen(palette().color(QPalette::Text));

    QRect textRect = p.fontMetrics().boundingRect(detailButtonText());
    textRect.setWidth(textRect.width() + 15);
    textRect.setHeight(textRect.height() + 4);
    textRect.moveCenter(rect().center());

    p.drawText(textRect, Qt::AlignLeft | Qt::AlignVCenter, detailButtonText());

    if (checkable) {
        int arrowsize = 15;
        QStyleOption arrowOpt;
        arrowOpt.initFrom(this);
        QPalette pal = arrowOpt.palette;
        pal.setBrush(QPalette::All, QPalette::Text, QColor(0, 0, 0));
        arrowOpt.rect = QRect(size.width() - arrowsize - 6, height()/2-arrowsize/2, arrowsize, arrowsize);
        arrowOpt.palette = pal;
        style()->drawPrimitive(checked ? QStyle::PE_IndicatorArrowUp : QStyle::PE_IndicatorArrowDown, &arrowOpt, &p, this);
    }

    m_cachedPixmaps.insert(key, pixmap);
    return pixmap;
}

bool DetailsButton::PixmapCacheKey::operator<(const PixmapCacheKey& other) const
{
    if (size.width() < other.size.width())
        return true;
    else if (other.size.width() < size.width())
        return false;
    if (size.height() < other.size.height())
        return true;
    else if (other.size.height() < size.height())
        return false;
    if (!checkable && other.checkable)
        return true;
    else if (!other.checkable && checkable)
        return false;
    if (!checked && other.checked)
        return true;
    else if (!other.checked && checked)
        return false;
    if (!down && other.down)
        return true;
    else if (!other.down && down)
        return false;
    if (!hover && other.hover)
        return true;
    else if (!other.hover && hover)
        return false;
    if (!enabled && other.enabled)
        return true;
    else if (!other.enabled && enabled)
        return false;
    return false;
}

QString DetailsButton::detailButtonText() const
{
    if (m_collapseText.isEmpty() && m_expandText.isEmpty())
    {
        if (isCheckable())
            return tr("Details");
        else
            return tr("Details...");
    }
    else
    {
        QString text = isChecked()? m_collapseText : m_expandText;
        if (isCheckable())
            text += "...";
        return text;
    }
}

void DetailsButton::setDetailButtonText(const QString& expandText, const QString &collapseText)
{
    m_expandText = expandText;
    m_collapseText = collapseText;
    update();
    updateGeometry();
}
